/*=======================================================================================

 M24XXX: Stimuli for I2C Bus EEPROM Verilog Simulation Model

=========================================================================================

 This program is provided "as is" without warranty of any kind, either
 expressed or implied, including but not limited to, the implied warranty
 of merchantability and fitness for a particular purpose. The entire risk
 as to the quality and performance of the program is with you. Should the
 program prove defective, you assume the cost of all necessary servicing,
 repair or correction.
 
 Copyright 2012, STMicroelectronics Corporation, All Right Reserved.

=======================================================================================*/
/*=====================================
 Stimuli Definition File
=====================================*/

`include "M24XXX_Parameters.v"

//`define  tL_DIV2  650
//`define  tH_DIV2  650

//===========================
module Stimuli(SCL,SDA);
inout  SDA;
output SCL;

//---------------------------
wire SDA;
reg SCL,OE,sda_out;

//=====================================
assign SDA = OE ? 1'bz : sda_out;
//=====================================

///////////////////////////////////////
//Current Address Read Test pattern
///////////////////////////////////////
task Current_address_read;
input n;
input[6:0] DEV_SEL;             //device select code
//-----------------             
reg[7:0] rd_dat;                //save read out data
integer n,i,j;                  
//-----------------             
begin                           
    I2C_Start;                  
    Dev_Sel_Code_R(DEV_SEL);    //send device select code with bit0=1, read
    Wait_Ack;                   
    Read_data(n);               //receive the read out data byte and ACK or NACK
    I2C_Stop;
end
endtask

///////////////////////////////////////
//Random Address Read Test pattern
///////////////////////////////////////
task Random_address_read;
input n;
input[6:0] DEV_SEL;             //device select code
input[`Address_Bits-1:0] address;
//-----------------
reg[7:0] rd_dat;                //save read out data
integer n,i,j;
//-----------------
begin
    I2C_Start;
    Dev_Sel_Code_W(DEV_SEL);    //send first device select code with bit0=0
    Wait_Ack; 
`ifndef MEM_1K_to_16K                  
    Write_add_h(address);       //send high byte of address
    Wait_Ack;   
`endif                
    Write_add_l(address);       //send low byte of address
    Wait_Ack;                   
    I2C_Start;                  //i2c bus re-start
    Dev_Sel_Code_R(DEV_SEL);    //send second device select code with bit0=1
    Wait_Ack;                   
    Read_data(n);               //receive the read out data byte and ACK or NACK
    I2C_Stop;
end
endtask

////////////////////////////////////////////////////////////////
//Write Operation Test pattern
////////////////////////////////////////////////////////////////
task Write_in_data;
input n;                        //the number of write data 
input[6:0] DEV_SEL;             //device select code
input[`Address_Bits-1:0] address;
input[7:0] data;                //write in data
//-----------------
reg[7:0] dat;
integer n,i,j;
//-----------------
begin
    I2C_Start;
    Dev_Sel_Code_W(DEV_SEL);    //send first device select code with bit0=0
    Wait_Ack;
`ifndef MEM_1K_to_16K                  
    Write_add_h(address);       //send high byte of address
    Wait_Ack;
`endif                
    Write_add_l(address);       //send low byte of address
    Wait_Ack;
    Write_data(n,data);         //send data byte to program
    I2C_Stop;
end
endtask

////////////////////////////////////////////////////////////////
//Read Lock status Operation Test pattern
////////////////////////////////////////////////////////////////
task Read_lock_status;
input[6:0] DEV_SEL;             //device select code
input[7:0] data;                //write in data
//-----------------
reg[7:0] dat;
integer n,i,j;
//-----------------
begin
    I2C_Start;
    Dev_Sel_Code_W(DEV_SEL);    //send first device select code with bit0=0
    Wait_Ack;
    Write_1data(data);       //send 1 byte of data
    Wait_Ack;
    I2C_Stop;
end
endtask

task Write_lock_status;
input[2:0] dev_id;
begin
    Write_in_data(1,{dev_id,4'b1000},`Address_Bits'd1024,8'hA2);
end
endtask

//================================
task Start_Stop;
begin
    I2C_Start;
    I2C_Stop;
end
endtask

//================================
//Send I2C Start Condition
//================================
task I2C_Start;
begin
    OE = 1'b0;                  //SDA as output
    SCL = 1'b0;
    sda_out = 1'b1;             //output high
    #1000;
    SCL = 1'b1;
    //#599;
    #(`tH_DIV2);                //I2C Start Condition Set Up Time
    sda_out = 1'b0;
    //#599;
    #(`tH_DIV2);                //I2C Start Condition Hold Time
    SCL = 1'b0;
    #(`tL_DIV2);
end
endtask

//================================
//Send I2C STOP Condition
//================================
task I2C_Stop;
begin
    OE = 1'b0;                  //SDA as output
    SCL = 1'b0;
    sda_out = 1'b0;             //output low
    #(`tL_DIV2);
    SCL = 1'b1;
    //#599;
    #(`tH_DIV2);                //I2C Stop Condition Set Up Time
    sda_out = 1'b1;
    #(`tH_DIV2);
    #(1000);
end
endtask

//================================
//Wait ACK from memory
//================================
task Wait_Ack;
begin
    OE = 1'b1;                  //SDA as input
    SCL = 1'b0;
    #(`tL_DIV2);
    SCL = 1'b1;
    #(2*`tH_DIV2);              //Clock High Pulse Width
    SCL = 1'b0;
    #(`tL_DIV2);
end
endtask

//===============================================
//Send Device Select Code with bit0 = 1(read)
//===============================================
task Dev_Sel_Code_R;
input[6:0] DEV_SEL;
//-----------------
integer i;
reg[7:0] dat;
//-----------------
begin
    dat = {DEV_SEL,1'b1};
    for(i=0;i<=7;i=i+1)
    begin
        OE = 1'b0;              //SDA as output
        SCL = 1'b0;
        sda_out = dat[7-i];
        //#99;
        #(`tL_DIV2);            //Data In Set Up Time
        SCL = 1'b1;
        #(2*`tH_DIV2);          //Clock High Pulse Width
        SCL = 1'b0;
        #(`tL_DIV2);            //Data In Hold Time
    end
end
endtask

//===============================================
//Send Device Select Code with bit0 = 0(write)
//===============================================
task Dev_Sel_Code_W;
input[6:0] DEV_SEL;
//-----------------
integer i;
reg[7:0] dat;
//-----------------
begin
    dat = {DEV_SEL,1'b0};
    for(i=0;i<=7;i=i+1)
    begin
        OE = 1'b0;              //SDA as output
        SCL = 1'b0;
        sda_out = dat[7-i];
        #(`tL_DIV2);            //Data In Set Up Time
        SCL = 1'b1;
        #(2*`tH_DIV2);          //Clock High Pulse Width
        SCL = 1'b0;
        #(`tL_DIV2);            //Data In Hold Time
    end
end
endtask

//===========================
//Send high byte of address
//===========================
`ifndef MEM_1K_to_16K
task Write_add_h;
input[`Address_Bits-1:0] address;
//-----------------
integer i;
reg[7:0] dat;
//-----------------
begin
    dat = {3'b000,address[`Address_Bits-1:8]};
    for(i=0;i<=7;i=i+1)
    begin
        OE = 1'b0;              //SDA as output
        SCL = 1'b0;
        sda_out = dat[7-i];
        #(`tL_DIV2);            //Data In Set Up Time
        SCL = 1'b1;
        #(2*`tH_DIV2);          //Clock High Pulse Width
        SCL = 1'b0;
        #(`tL_DIV2);            //Data In Hold Time
    end
end
endtask
`endif
//===========================
//Send low byte of address
//===========================
task Write_add_l;
input[`Address_Bits-1:0] address;
//-----------------
integer i;
reg[7:0] dat;
//-----------------
begin
    dat = address[7:0];
    for(i=0;i<=7;i=i+1)
    begin
        OE = 1'b0;              //SDA as output
        SCL = 1'b0;
        sda_out = dat[7-i];
        #(`tL_DIV2);            //Data In Set Up Time
        SCL = 1'b1;
        #(2*`tH_DIV2);          //Clock High Pulse Width
        SCL = 1'b0;
        #(`tL_DIV2);            //Data In Hold Time
    end
end
endtask

//===========================
//Send 1 byte of data
//===========================
task Write_1data;
input[7:0] data;
//-----------------
integer i;
reg[7:0] dat;
//-----------------
begin
    dat = data;
    for(i=0;i<=7;i=i+1)
    begin
        OE = 1'b0;              //SDA as output
        SCL = 1'b0;
        sda_out = dat[7-i];
        #(`tL_DIV2);            //Data In Set Up Time
        SCL = 1'b1;
        #(2*`tH_DIV2);          //Clock High Pulse Width
        SCL = 1'b0;
        #(`tL_DIV2);            //Data In Hold Time
    end
end
endtask

//===============================================
//Receive the read out data byte and ACK or NACK
//===============================================
task Read_data;
input n;
//-----------------
integer i,j,n;
reg[7:0] rd_dat;
//-----------------
begin
    for(i=0;i<=n-1;i=i+1)
    begin
        for(j=0;j<=7;j=j+1)
        begin
            OE = 1'b1;                  //SDA as input
            SCL = 1'b0;
            #(`tL_DIV2);
            SCL = 1'b1;
            rd_dat = {rd_dat[6:0],SDA}; //save the read out serial data
            #(2*`tH_DIV2);              //Clock High Pulse Width
            SCL = 1'b0;
            #(`tL_DIV2);
        end
        $display("%t,address=[%h],data=[%h]",$time,U_M24XXX.memory_address,rd_dat);
    //----------------------------------//Send ACK to memory
        OE = 1'b0;                      //SDA as output
        SCL = 1'b0;
        if(i<n-1)    sda_out = 1'b0;    //ACK the data byte read from memory
        if(i==(n-1)) sda_out = 1'b1;    //NACK the last data byte read from memory
        #(`tL_DIV2);
        SCL = 1'b1;
        #(2*`tH_DIV2);                  //Clock High Pulse Width
        SCL = 1'b0;
        #(`tL_DIV2);
    end
end
endtask

//================================
//send data byte to program
//================================
task Write_data;
input n;
input[7:0] data;
//-----------------
reg[7:0] dat;
integer i,j,n;
//-----------------
begin
    dat = data[7:0];
    for(i=0;i<=n-1;i=i+1)
    begin
        for(j=0;j<=7;j=j+1)
        begin
            OE = 1'b0;                  //SDA as output
            SCL = 1'b0;
            sda_out = dat[7-j];
            #(`tL_DIV2);                //Data In Set Up Time
            SCL = 1'b1;
            #(2*`tH_DIV2);              //Clock High Pulse Width
            SCL = 1'b0;
            #(`tL_DIV2);                //Data In Hold Time
        end
    //----------------------------------//Wait ACK from memory
        OE = 1'b1;                      //SDA as input
        SCL = 1'b0;
        #(`tL_DIV2);
        SCL = 1'b1;
        #(2*`tH_DIV2);                  //Clock High Pulse Width
        SCL = 1'b0;
        #(`tL_DIV2);
    end
end
endtask
/////////////////////////////////////////////////
endmodule
